<?php
session_start();
// ==============================================================================
// CONFIGURAÇÕES E DEBUG
// ==============================================================================
ini_set('display_errors', 0);
error_reporting(E_ALL);

function logMsg($msg) {
    file_put_contents('debug_pix.txt', date('d/m/Y H:i:s') . " - " . print_r($msg, true) . PHP_EOL, FILE_APPEND);
}

// --- TRAVA ANTI-DUPLICAÇÃO (NOVO) ---
// Se o usuário clicou em pagar há menos de 10 segundos, bloqueia
if (isset($_SESSION['last_payment_click']) && (time() - $_SESSION['last_payment_click'] < 10)) {
    // Redireciona para o último pedido gerado (se houver) ou para a home
    if (isset($_SESSION['last_order_id'])) {
        header("Location: pagamento_pix.php?pedido_id=" . $_SESSION['last_order_id']);
        exit;
    }
}
$_SESSION['last_payment_click'] = time();

logMsg("=== NOVA TRANSAÇÃO INICIADA ===");

// 1. CONEXÃO BLINDADA
require_once __DIR__ . '/../config.php';

// 2. BUSCA CONFIGURAÇÃO DO GATEWAY
$stmt = $pdo->prepare("SELECT * FROM config_gateway WHERE id = 1");
$stmt->execute();
$gateway = $stmt->fetch();

if (!$gateway || empty($gateway['access_token'])) { die("Erro: API Key não configurada."); }

$api_token = $gateway['access_token'];
$api_url = "https://api.witetec.net/transactions"; 

// 3. RECEBE DADOS
if ($_SERVER['REQUEST_METHOD'] !== 'POST') { header("Location: index.php"); exit; }

$nome = htmlspecialchars($_POST['nome'] ?? 'Cliente', ENT_QUOTES, 'UTF-8');
$email = filter_input(INPUT_POST, 'email', FILTER_VALIDATE_EMAIL);
$cpf = preg_replace('/[^0-9]/', '', $_POST['cpf'] ?? '');
$whatsapp = preg_replace('/[^0-9]/', '', $_POST['whatsapp'] ?? '');
$produto_id = filter_input(INPUT_POST, 'product_id', FILTER_VALIDATE_INT);

if (!$email || !$produto_id || !$cpf) { die("Dados inválidos. Preencha tudo."); }

try {
    $pdo->beginTransaction();

    // 4. BUSCA PRODUTO
    $stmt = $pdo->prepare("SELECT * FROM produtos WHERE id = ?");
    $stmt->execute([$produto_id]);
    $prod = $stmt->fetch();
    if (!$prod) throw new Exception("Produto não encontrado.");

    $amount_cents = (int)($prod['preco_atual'] * 100); 

    // 5. GESTÃO DE USUÁRIO
    $stmt = $pdo->prepare("SELECT id FROM usuarios WHERE email = ?");
    $stmt->execute([$email]);
    $usuario = $stmt->fetch();
    
    if ($usuario) {
        $usuario_id = $usuario['id'];
        $pdo->prepare("UPDATE usuarios SET nome=?, cpf=?, whatsapp=? WHERE id=?")->execute([$nome, $cpf, $whatsapp, $usuario_id]);
    } else {
        $senha = password_hash('tiktok', PASSWORD_DEFAULT);
        $pdo->prepare("INSERT INTO usuarios (nome, email, cpf, whatsapp, senha, tipo, data_criacao) VALUES (?, ?, ?, ?, ?, 'cliente', NOW())")
            ->execute([$nome, $email, $cpf, $whatsapp, $senha]);
        $usuario_id = $pdo->lastInsertId();
    }

    // 6. CRIA O PEDIDO
    $pdo->prepare("INSERT INTO pedidos (usuario_id, produto_id, valor_total, status, forma_pagamento, data_pedido) VALUES (?, ?, ?, 'pendente', 'pix', NOW())")
        ->execute([$usuario_id, $produto_id, $prod['preco_atual']]);
    $pedido_id = $pdo->lastInsertId();
    
    // Salva na sessão para evitar duplicação no refresh
    $_SESSION['last_order_id'] = $pedido_id;

    logMsg("Pedido #$pedido_id criado. Enviando para API...");

    // 7. CHAMADA À API
    $payload = [
        "amount" => $amount_cents,
        "method" => "PIX",
        "metadata" => [ "sellerExternalRef" => (string)$pedido_id ],
        "customer" => [
            "name" => $nome,
            "email" => $email,
            "phone" => $whatsapp,
            "documentType" => "CPF",
            "document" => $cpf
        ],
        "items" => [
            [
                "title" => $prod['nome'],
                "amount" => $amount_cents,
                "quantity" => 1,
                "tangible" => false,
                "externalRef" => (string)$prod['id']
            ]
        ]
    ];

    $curl = curl_init();
    curl_setopt_array($curl, [
        CURLOPT_URL => $api_url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_ENCODING => "",
        CURLOPT_MAXREDIRS => 10,
        CURLOPT_TIMEOUT => 30,
        CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
        CURLOPT_CUSTOMREQUEST => "POST",
        CURLOPT_POSTFIELDS => json_encode($payload),
        CURLOPT_HTTPHEADER => [
            "Content-Type: application/json",
            "x-api-key: " . $api_token
        ],
    ]);

    $response = curl_exec($curl);
    $http_code = curl_getinfo($curl, CURLINFO_HTTP_CODE);
    $err = curl_error($curl);
    curl_close($curl);

    if ($err) throw new Exception("Erro cURL: " . $err);

    $retorno = json_decode($response, true);

    if ($http_code !== 201 || !isset($retorno['status']) || $retorno['status'] !== true) {
        $msg_erro = $retorno['message'] ?? 'Erro na API';
        logMsg("ERRO API: $response");
        throw new Exception("Gateway recusou: $msg_erro");
    }

    // ==========================================================================
    // 8. FILTRO INTELIGENTE DE PIX (AQUI É O SEGREDO)
    // ==========================================================================
    $transaction_id = $retorno['data']['id'];
    $raw_qrcode = $retorno['data']['pix']['qrcode'] ?? '';
    $raw_copyPaste = $retorno['data']['pix']['copyPaste'] ?? '';
    $raw_url = $retorno['data']['pix']['qrcodeUrl'] ?? '';

    $pix_limpo = '';

    // Regra: Só aceita se começar com "000201" (Padrão Oficial do Pix)
    if (!empty($raw_copyPaste) && strpos($raw_copyPaste, '000201') === 0) {
        $pix_limpo = $raw_copyPaste;
    } elseif (!empty($raw_url) && strpos($raw_url, '000201') === 0) {
        $pix_limpo = $raw_url;
    } elseif (!empty($raw_qrcode) && strpos($raw_qrcode, '000201') === 0) {
        $pix_limpo = $raw_qrcode;
    } else {
        // Se nenhum começar com 000201, pega o copyPaste como fallback (mesmo que venha estranho, é a melhor aposta)
        $pix_limpo = $raw_copyPaste;
    }

    // Grava no log o que decidiu usar para debug futuro
    logMsg("Pix escolhido para o banco: " . substr($pix_limpo, 0, 20) . "...");

    $qr_code_img = $retorno['data']['pix']['qrcode_base64'] ?? '';

    // Atualiza pedido
    $pdo->prepare("UPDATE pedidos SET transaction_id=?, pix_copia_cola=?, qr_code_img=? WHERE id=?")
        ->execute([$transaction_id, $pix_limpo, $qr_code_img, $pedido_id]);

    $pdo->commit();

    // 9. REDIRECIONAMENTO
    header("Location: pagamento_pix.php?pedido_id=" . $pedido_id);
    exit;

} catch (Exception $e) {
    if ($pdo->inTransaction()) { $pdo->rollBack(); }
    logMsg("EXCEÇÃO: " . $e->getMessage());
    die("<div style='font-family:sans-serif;padding:30px;text-align:center;'><h1>Erro</h1><p>{$e->getMessage()}</p><button onclick='history.back()'>Voltar</button></div>");
}
?>